package eu.nomad_lab

import java.nio.file.Paths

import com.typesafe.config.{ ConfigFactory, Config }
import com.typesafe.scalalogging.StrictLogging
import eu.nomad_lab.QueueMessage.TreeParserRequest
import eu.nomad_lab.parsing_queue.TreeParser
import org.json4s.JsonAST.JValue
import scala.collection.mutable
import scala.io.Source

import com.rabbitmq.client._

object TreeParserInitilaizer extends StrictLogging {

  /**
   * The settings required to get the read and write queue
   */
  class Settings(config: Config) {
    // validate vs. reference.conf
    config.checkValid(ConfigFactory.defaultReference(), "simple-lib")
    val rabbitMQHost = config.getString("nomad_lab.parser_worker_rabbitmq.rabbitMQHost")
    val rabbitMQPort = config.getString("nomad_lab.parser_worker_rabbitmq.rabbitMQPort")
    val writeToExchange = config.getString("nomad_lab.parser_worker_rabbitmq.treeParserExchange")
    val writeQueue = config.getString("nomad_lab.parser_worker_rabbitmq.treeParserQueue")
    def toJValue: JValue = {
      import org.json4s.JsonDSL._
      (("rabbitMQHost" -> rabbitMQHost) ~
        ("writeToExchange" -> writeToExchange))
    }
  }

  val settings = new Settings(LocalEnv.defaultConfig)
  val parserCollection = parsers.AllParsers.defaultParserCollection
  val treeParser = new TreeParser(
    parserCollection = parserCollection
  )

  val usage = """treeparserinitializer [--help] [--verbose]
       [--file file1 [--file file2 ...]]
       [pathToArchive1 pathToArchive2...]"""

  /**
   * Main function of TreeParserInitilaizer
   *
   * @param args
   */
  def main(args: Array[String]): Unit = {
    val filesToScan = mutable.ListBuffer[String]()
    var list: List[String] = args.toList
    var verbose: Boolean = false
    while (!list.isEmpty) {
      val arg = list.head
      list = list.tail
      arg match {
        case "--help" | "-help" | "-h" =>
          println(usage)
          return
        case "--verbose" =>
          verbose = true
        case "--file" =>
          if (list.isEmpty) {
            println(s"Error: expected a file after --file, $usage")
            return
          }
          filesToScan ++= Source.fromFile(list.head).getLines
          list = list.tail
        case _ =>
          filesToScan += arg
      }
    }
    writeToQueue(filesToScan)
  }

  /**
   * Find the parsable files and parsers. Write this information for the single step parser
   */
  def writeToQueue(args: Seq[String]) = {
    val prodFactory: ConnectionFactory = new ConnectionFactory
    prodFactory.setHost(settings.rabbitMQHost)
    prodFactory.setPort(settings.rabbitMQPort.toInt)
    val prodConnection: Connection = prodFactory.newConnection
    val prodchannel: Channel = prodConnection.createChannel
    prodchannel.exchangeDeclare(settings.writeToExchange, "fanout")
    prodchannel.queueDeclare(settings.writeQueue, true, false, false, null)
    prodchannel.queueBind(settings.writeQueue, settings.writeToExchange, "")
    for (filePath <- args) {
      val message = TreeParserRequest.fromPath(Paths.get(filePath))
      val msgBytes = JsonSupport.writeUtf8(message)
      logger.info(s"Message: $message, bytes Array size: ${msgBytes.length}")
      try {
        prodchannel.basicPublish(settings.writeToExchange, "", null, msgBytes)
      } catch {
        case e: Exception =>
          logger.error(s"Error while publishing to the exchange.Message: ${e.getMessage}")
          System.exit(1)
      }
    }
    logger.info(s"Wrote to Exchange: ${settings.writeToExchange}")
    prodchannel.close
    prodConnection.close
  }
}
